#include "Utilities/Configuration/interface/Architecture.h"

#include "TrackerReco/ClusterShaper/interface/FitClusterParameters.h"

#include "TrackerReco/ClusterShaper/interface/ChannelsTouched.h"
#include "TrackerReco/ClusterShaper/interface/ChannelsOutlier.h"
#include "TrackerReco/ClusterShaper/interface/Transformations.h"
#include "TrackerReco/ClusterShaper/interface/LevenbergMarquardt.h"

#include <iostream>

extern geom_t geom;

/*****************************************************************************/
FitClusterParameters::FitClusterParameters() { }

/*****************************************************************************/
FitClusterParameters::~FitClusterParameters() { }

/*****************************************************************************/
double FitClusterParameters::function(double z, int rank)
{
 // Gauss
 if(rank == 0) return(2 * z*z/2);
 if(rank == 1) return(    z  );
          else return(    1. );
}

/*****************************************************************************/
void FitClusterParameters::fillAdc(int nchannel,channel_t *channel,
                    int *npixel ,vector<pixel_t>* pixel,
                    double endpoint[2][2])
{
 // Clear adc
 for(int i=0; i<*npixel; i++)
 {
  (*pixel)[i].adc  = 0;
  (*pixel)[i].sign = 1;
 }

 // Try to match channels
 for(int j=0; j<nchannel; j++)
 {
  int found = 0;

  for(int i=0; i<*npixel; i++)
  {
   if(channel[j].position[0] == (*pixel)[i].position[0] &&
      channel[j].position[1] == (*pixel)[i].position[1])
   { (*pixel)[i].adc = channel[j].adc; found=1; }
  }

  if(!found) // This channel is not touched by the line, a new one, add it!
  {
   pixel_t pix;

   for(int k=0; k<2; k++)
    pix.position[k] = channel[j].position[k];

   pix.adc = channel[j].adc;

   // Calculate negative distance
   ChannelsOutlier theChannelsOutlier;
   theChannelsOutlier.negativeDistance(&pix, endpoint);
   pixel->push_back(pix);

   (*npixel)++;
  }
 }
}

/*****************************************************************************/
void FitClusterParameters::preparePixels(double endpoint[][2])
{
 // Reset pixels
 pixel.clear();

 // Calculate path
 ChannelsTouched theChannelsTouched;
 npixel = theChannelsTouched.getPath(endpoint, &pixel);

 for(int i=0; i<npixel; i++)
 {
  pixel[i].endpoint = 0;

  for(int j=0; j<2; j++)
   if(pixel[i].position[0] == (int)endpoint[j][0] &&
      pixel[i].position[1] == (int)endpoint[j][1])
    pixel[i].endpoint = 1;
  }

 // Fill adcs
 fillAdc(nchannel,&channel[0], &npixel,&pixel, endpoint);
}

/*****************************************************************************/
void FitClusterParameters::calculateTotalLength3(double point[][2], double *length2d, double *length3d)
{
 int k; 
 double dpos[2];
 
 *length2d = 0.; // Total 2d length!!
 *length3d = 0.; // Total 3d length!!
 
 for(k=0; k<2; k++)
 {
  dpos[k] = (point[1][k] - point[0][k]);
  *length2d += sqr(dpos[k] * geom.pitch[k]);
  
  dpos[k] = (point[1][k] - point[0][k]) +
            (!geom.flipped ? 1 : -1) * (-geom.shift[k]); // undo Lorentz
  *length3d += sqr(dpos[k] * geom.pitch[k]);
 }
 
 *length3d += sqr(geom.thickness); // thickness
 
 *length2d = sqrt(*length2d);
 *length3d = sqrt(*length3d);
}

/*****************************************************************************/
void FitClusterParameters::getFunction(double cluster[], double *chi2)
{
 // Get cluster endpoints
 double endpoint[2][2];
 Transformations theTransformations;
 theTransformations.transformClusterToEndpoint(cluster, endpoint);

 preparePixels(endpoint);

 // Clear
 *chi2 = 0.;

 // Sum for all the touched channels
 for(int i=0; i<npixel; i++)
 {
  double val = cluster[4] * pixel[i].sign * pixel[i].length;
  *chi2 += function(pixel[i].adc - val, 0);
 }
}

/*****************************************************************************/
double FitClusterParameters::crossProduct(double a[], double b[])
{
 return(a[0]*b[1] - a[1]*b[0]);
}

/*****************************************************************************/
double FitClusterParameters::scalarProduct2(int n, double a[],double b[])
{
 double s=0.;

 for(int k=0; k<n; k++) s += a[k]*b[k];

 return(s);
}

/*****************************************************************************/
void FitClusterParameters::collectDerivativesCluster
(double par[],pixel_t *pixel, double length2d, double length3d,
 double der[], double *val)
{
 int j, sum[2];
 double v[2], s, f, delta[2];

 v[0] = cos(par[3]);
 v[1] = sin(par[3]);

 for(j=0; j<2; j++)
  delta[j] = (!geom.flipped ? 1 : -1) * (-geom.shift[j]) * geom.pitch[j];
 f = scalarProduct2(2,delta,v);

 // Center x and y position, angle
 // dl/da
 der[0] = 0.; der[1] = 0.; der[3] = 0.;
 for(j=0; j<pixel->nc; j++)
 {
  s = fabs(scalarProduct2(2, v,pixel->normal[j]));

  der[3] += crossProduct (  pixel->dx[j],pixel->normal[j])/s;
 }

 der[3] = pixel->sign * par[4] *
  (pixel->length * crossProduct(v,delta)/length3d +
   der[3] * length3d/length2d);

 // Calculate separately der[0] and der[1] !!!!
 sum[0]=0; sum[1]=0;
 for(j=0; j<pixel->nc; j++)
 {
  if(pixel->normal[j][0] > 0.) sum[0]++;
  if(pixel->normal[j][0] < 0.) sum[0]--;

  if(pixel->normal[j][1] > 0.) sum[1]++;
  if(pixel->normal[j][1] < 0.) sum[1]--;
 }

 // dl/dx[0]
 if(sum[0] != 0)
  der[0] = pixel->sign * par[4] * sum[0] / fabs(v[0]) * geom.pitch[0] *
            length3d/length2d;
 else der[0] = 0.;

 // dl/dy[0]
 if(sum[1] != 0)
  der[1] = pixel->sign * par[4] * sum[1] / fabs(v[1]) * geom.pitch[1] *
            length3d/length2d;
 else der[1] = 0.;

 // half-length
 der[2] = par[4] *
  (pixel->endpoint*length3d/length2d +
   2*pixel->sign * pixel->length*((length2d + f)/length3d*length2d - length3d)/sqr(length2d));

 // energy loss
 der[4] = pixel->sign * pixel->length;

 // value
 *val = par[4] * pixel->sign * pixel->length;
}

/*****************************************************************************/
void FitClusterParameters::getDerivatives
  (double cluster[], double beta[],Matrix<double>* alpha)
{
 double der[Npar],val, chi2=0., endpoint[2][2];
 double length2d,length3d;

 Transformations theTransformations;
 theTransformations.transformClusterToEndpoint(cluster, endpoint);

 // Clear
 for(int k=0; k<Npar; k++)
 {
  beta[k] = 0.;
  for(int l=0; l<Npar; l++) (*alpha)(k,l) = 0.;
 }

 calculateTotalLength3(endpoint, &length2d,&length3d);

 // For all the pixels
 for(int i=0; i<npixel; i++)
 {
  // Get derivatives
  collectDerivativesCluster(cluster,&pixel[i], length2d,length3d, der,&val);

  chi2 += function(pixel[i].adc - val, 0);

  // Fill beta and alpha (for cluster)
  for(int k=0; k<Npar; k++)
  {
   beta[k] += function(pixel[i].adc -  val, 1) * der[k];
   for(int l=0; l<Npar; l++)
    (*alpha)(k,l) += function(pixel[i].adc -  val, 2) * der[k] * der[l];
  }
 }
}

/*****************************************************************************/
void FitClusterParameters::fit(int ncha,int val[][3], double endpoint[][2], double track[][Npar+1])
{
 // Allocate
 pixel.reserve(10);
 channel.reserve(10);

 // Copy
 channel_t cha;
 nchannel = ncha;
 for(int i=0; i<nchannel; i++)
 {
  cha.position[0] = val[i][0];
  cha.position[1] = val[i][1];
  cha.adc         = val[i][2];

  channel.push_back(cha);
 }

 Transformations    theTransformations;
 LevenbergMarquardt<FitClusterParameters> theLevenbergMarquardt;

 // Try both direction, by flipping cluster direction / angle
 for(int entry=0; entry<2; entry++)
 {
  double cluster[Npar+1], err[Npar];

  if(debug())
   cerr << "  [ClusterShaper]  entry " << entry << endl;
  theTransformations.transformEndpointToCluster(endpoint, cluster);
  cluster[3] += entry*M_PI;
 
  int nstep;
  cluster[Npar] =
	   theLevenbergMarquardt.minimize(this, Npar, cluster, err, &nstep);

  // Attempt to correct z0 = 1.5 - 2.5, increase pathlength 
//  cluster[2] += 18e-4; // 18 um

  track[entry][Npar] = cluster[Npar];
  theTransformations.transformClusterToTrack(cluster,track[entry]);
 }
}

